function [Rg_evol,Rg_evol_vector,Rg_end,time_array] = RgEvolution(trajfile,filename,identifier,magnif,xposure)

%function [Rg_evol,Rg_evol_vector,Rg_end,time_array] = RgEvolution(trajfile,filename,identifier,magnif,xposure)
%
%This function is designed to calculate the radius of gyration v. time for each
%particle in a population.  This results in a plot of the Rg evolution.
%
%INPUTS:    TRAJFILE:   Dataset of the format traj, columns 1, 2, 7, and 8
%                       are used.  Any matrix with the following columns of 
%                       information can be used.
%           TRAJ        This array contains all the information learned
%                       about all the particles, stored in columns. A
%                       single trajectory may occupy many sequential rows.
%                       A single row details information about a feature in
%                       a particular frame, or that feature's trajectory
%                       from the previous frame to that frame.
%           TRAJ(:,1)   The x centroid positions, in pixels
%           TRAJ(:,2)   The y centroid positions, in pixels
%           TRAJ(:,7)   The unique particle ID of this particle
%           TRAJ(:,8)   How many frames this particle has been tracked so
%                       far 
%           FILENAME:   Name of the original movie file that was analyzed
%                       for tracking and from which TRAJ was formed,
%                       with or without filepath.  Either way, output .dat
%                       file will be named with 'filename time_Rg.dat' in
%                       the current folder.
%           IDENTIFIER: A string chosen to add to time and PID labels for
%                       clarification from other data.
%           MAGNIF:     Magnification of the microscope, for calculating
%                       distance in microns instead of pixels.
%           XPOSURE:    Kinetic cycle time (from beginning of one frame to
%                       the next) of the movie.

%OUTPUTS:   RG_EVOL:    An array with 'number of particles' columns and
%                       'longest trajectory - 1' rows, with radius of gyration
%                       evolution for each particle.  Since trajectories
%                       are of various lengths, only the longest
%                       trajectory has all values filled in, the rest end
%                       with NaNs in the remaining cells, which makes 
%                       plotting convenient.
%           TIME_ARRAY: A 1D array with time in seconds.
%           RG_EVOL_VECTOR: A 1D array with the same information as
%                       RG_EVOL, except that it's in a single column
%           RG_END:     A 1D array with the final Rg value from each
%                       molecule.
%           .DAT:       A .dat file with the first row = transposed time_array
%                       and each row after that a particles Rg evolution.
%                       The first column is labeled Time_pt_identifier_time
%                       and identifier_PIDnumber.  The file is named with
%                       the same root as filename.
%
%For example: traj has three particles, the longest is around for 6 frames
%[Rg_evol,time_array] = RgEvolution(traj,file,'10kDa_23.4',magnif,xposure);
%Rg_evol:   
%0.441318126	0.27429858	0.04663554
%0.45992436     0.295499318	0.047937444
%0.649460615	NaN         0.055753191
%0.7327759      NaN         NaN
%0.71901843     NaN         NaN
%
%Time_array: 
%0.11118
%0.16677
%0.22236
%0.27795
%0.33354
%
%23.4C time_Rg.dat
%Time_pt_10kDa_23.4 0.11118     0.16677     0.22236     0.27795     0.33354
%10kDa_23.4_PID1    0.44132     0.45992     0.64946     0.73278      0.7719
%10kDa_23.4_PID2    0.2743      0.2955      NaN         NaN         NaN            
%10kDa_23.4_PID3    0.046636    0.047937    0.055753    NaN         NaN  
%
%
%Written by Lindsay Elliott, ND and UIUC, May 6, 2010

   
%Switch units from CCD pixels to microns using the pixel size, 16 um a
%side, of my camera and the magnification the microscope setup.
trajfile(:,1) = trajfile(:,1).*16./magnif; 
trajfile(:,2) = trajfile(:,2).*16./magnif;

%Make time array in seconds, xposure is the total kinetic cycle time of the
%acquisition (exposure time plus any delays)
time_array = xposure*(2:max(trajfile(:,8))+1)'/1000;

%Determine all the particle ID's used. 
PID=unique(trajfile(:,7));
PID=setdiff(PID,0);

%Initialize arrays
Rg_evol = NaN(max(trajfile(:,8)),length(PID));
Rg_evol_vector = [];
Rg_end = zeros(length(PID),1);
T = zeros(2,2);

%Loop through all particles
for i = 1:length(PID) 
    %Save the rows (all columns) in trajfile where PID = j as Ppos
    Ppos_Rg=trajfile(:,7)==PID(i);        

    %Save x and y position data for this particle in separate arrays
    x = trajfile(Ppos_Rg,1);
    y = trajfile(Ppos_Rg,2);
    
    %Length cutoffs, if desired
    if length(x) >=1 % && length(x) < 150     
        %Loop through each trajectory
        for j = 2:length(x)
            %Save x and y position data for this particle in separate arrays
            x_seg = x(1:j);
            y_seg = y(1:j);
            
            %Calculate radius of gyration
            %Populate the 2x2 tensor T for calculating the Rg     
            T(1,1) = mean(x_seg.^2) - mean(x_seg).^2;
            T(1,2) = mean(x_seg.*y_seg) - mean(x_seg).*mean(y_seg);
            T(2,1) = T(1,2);
            T(2,2) = mean(y_seg.^2) - mean(y_seg).^2;
            
            %Determine the eigenvalues of T and calculate Rg in microns
            d = eig(T);
            R1sq = d(1);
            R2sq = d(2);
            Rg_evol(j-1,i) = (R1sq + R2sq).^(0.5); 
            % Make a 1-column array with all the Rg values calculated for all
            % the molecules
            Rg_evol_vector = [Rg_evol_vector; (R1sq + R2sq).^(0.5)]; 
            % Make a 1-column array with the final Rg value for each
            % molecule
            Rg_end(i) = max((R1sq + R2sq).^(0.5));
        end
    end
end

%Write time and Rg data to a .dat file with labels for use in Igor or other
%data analysis software.  First row is time, each row after that is a
%particle, labels in the first column (Time_pt, PID1, PID2, etc).
[~, name, ~, ~] = fileparts(filename);
output_file_param = [name, ' time_Rg.dat'] ;
fid = fopen(output_file_param, 'at','native') ;
time_list = num2str(time_array', 5);
fprintf(fid, 'Time_pt_%s %s\n', identifier, time_list) ;
for param = 1:length(PID) 
    data = num2str(Rg_evol(:,param)' , 5) ;
    fprintf(fid, '%s_PID%d ', identifier, param) ;
    fprintf(fid, '%s\n', data) ;
end
fclose(fid) ;










